﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;

    using Newtonsoft.Json;

    using Shared.DataFilters;

    using Shared.Library.Enums;
    using Utilities;
    using Hims.Domain.Helpers;
    using Shared.UserModels.EncounterTemplates;
    using Hims.Shared.EntityModels;
    using Hims.Api.Models;

    /// <inheritdoc />
    [Authorize]
    [Route("api/encounter-template")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class EncounterTemplateController : BaseController
    {
        /// <summary>
        /// The encounter template services.
        /// </summary>
        private readonly IEncounterTemplateService encounterTemplateService;

        // <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;
        /// <inheritdoc />
        public EncounterTemplateController(IEncounterTemplateService encounterTemplateService, IAuditLogService auditLogService)
        {
            this.encounterTemplateService = encounterTemplateService;

            this.auditLogServices = auditLogService;
        }

        /// <summary>
        /// Adds the or edit template asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("modify-template")]
        public async Task<ActionResult> AddOrEditTemplateAsync([FromBody] EncounterTemplateModel model, [FromHeader] LocationHeader header)
        {
            model = (EncounterTemplateModel)EmptyFilter.Handler(model);
            var response = await this.encounterTemplateService.InsertOrEditTemplateAsync(model);
            try
            {
                if (response > 0)
                {
                    var auditLog = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.EncounterTemplate,
                        LogDate = DateTime.Now,
                        LogFrom = (short)model.LoginRoleId,
                        LocationId = Convert.ToInt32(header.LocationId),
                        LogDescription = $"<b>{model.CreatedByName}</b>  {(model.TemplateHeaderId == 0? "Added" : "Updated")} <b>Encounter Template</b> of <b>{model.ModuleName}</b> successfully.",

                    };
                    await this.auditLogServices.LogAsync(auditLog);
                }
            }
            catch(Exception ex)
            {
                throw ex;
            }
            return this.Success(response);
        }

           

        /// <summary>
        /// Fetches all templates asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-template")]
        public async Task<ActionResult> FetchAllTemplatesAsync([FromBody] EncounterTemplateModel model)
        {
            model ??= new EncounterTemplateModel();
            var response = await this.encounterTemplateService.FetchTemplatesAsync(model);
            return this.Success(response);
        }
    }
}